﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Drawing;
using System.ComponentModel;

namespace NetVision {

    /// <summary>
    /// ノードコントローラ
    /// </summary>
    public class NodeControl : AreaControl {


        /// <summary>
        /// アクティブノードの表示
        /// </summary>
        [Category("子コントロール")]
        public Label ActiveNodeControl { get; set; }

        /// <summary>
        /// スリープノードの表示
        /// </summary>
        [Category("子コントロール")]
        public Label SleepNodeControl { get; set; }

        /// <summary>
        /// 全ノードの表示
        /// </summary>
        [Category("子コントロール")]
        public Label AllNodeControl { get; set; }


        /// <summary>
        /// アクティブノードの数
        /// </summary>
        public int ActiveNodesCount {
            get {
                int i = 0;
                foreach (CardBox card in Cards) { if (card.Active) { i++; } }
                return i;
            }
        }

        /// <summary>
        /// スリープノードの数
        /// </summary>
        public int SleepNodesCount {
            get {
                int i = 0;
                foreach (CardBox card in Cards) { if (!card.Active) { i++; } }
                return i;
            }
        }

        /// <summary>
        /// ノードに追加
        /// </summary>
        /// <param name="card">追加するカード</param>
        public override void Add(CardBox card) {
            card.Area = AreaType.Nodes;
            card.FaceSide = false;
            card.Public = false;
            Cards.Add(card);
            Controls.Add(card);
            card.BringToFront();
            card.Location = new Point(10, Count * 15 + 30);
            card.Redraw();

            RedrawCards();
        }


        /// <summary>
        /// ノードの除外
        /// </summary>
        /// <param name="card">除外するノード</param>
        public override void Remove( CardBox card ) {
            Cards.Remove(card);
            Controls.Remove(card);
            RedrawCards();
        }

        /// <summary>
        /// ノードのアクティブ/スリープを切り替える
        /// </summary>
        /// <param name="card">切り替えるノードカード</param>
        /// <param name="active">アクティブか、スリープか</param>
        public void ChangeNode( CardBox card, bool active ) {
            if (!Cards.Contains(card)) { return; }
            card.Active = active;
            RedrawCards();
        }


        /// <summary>
        /// コントロールの再描画
        /// </summary>
        public override void RedrawCards() {
            AllNodeControl.Text = Count.ToString();
            ActiveNodeControl.Text = ActiveNodesCount.ToString();
            SleepNodeControl.Text = SleepNodesCount.ToString();
            AllNodeControl.BringToFront();
            ActiveNodeControl.BringToFront();
            SleepNodeControl.BringToFront();
        }


        /// <summary>
        /// 空き地を取得
        /// </summary>
        /// <returns>空き地の仮装座標</returns>
        public PointF GetEmptyLocation(CardBox card){
            int margin = 30;
            for (int j = 0; j < 2; j++) {
                int i = (j == 0) ? Count : 0;
                for (; i < 10; i++) {
                    Point xy = new Point(Size.Width / 2 - card.Size.Width / 2 + j * 20 - 10, ( Size.Height - card.Size.Height ) / 10 * i + margin);
                    PointF vp = card.GetVirtualPoint(xy, Size);
                    bool onthe = false;
                    foreach (CardBox nd in Cards) {
                        if (Math.Abs(nd.VirtualPoint.X - vp.X) < 0.05 && Math.Abs(nd.VirtualPoint.Y - vp.Y) < 0.05) { onthe = true; }
                    }
                    if (!onthe) { return vp; }
                }
            }
            return card.GetVirtualPoint(new Point(margin, margin), Size);
        }

    }
}
