﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.IO;
using System.Text;
using System.Windows.Forms;

namespace NetVision {

    public partial class CardBox : UserControl {
        public CardBox() {
            InitializeComponent();
        }

        #region 静的プロパティ

        /// <summary>
        /// 裏向きアクティブ画像
        /// </summary>
        public static Image BackImage { get; set; }

        /// <summary>
        /// 裏向きイメージのキャッシュ
        /// </summary>
        public static Image[] CachedBackImages = new Image[4];

        /// <summary>
        /// 汎用アクティブ画像
        /// </summary>
        public static Image GenericImage { get; set; }

        #endregion

        #region プロパティ

        /// <summary>
        /// カードID
        /// </summary>
        public int Id { get; set; }

        private CardInfo m_card = new CardInfo();
        /// <summary>
        /// カード情報
        /// </summary>
        public CardInfo Info { 
            get { return m_card; }
            set { LoadCard(value); }
        }

        /// <summary>
        /// カードの所有者（本来のプレイヤー）
        /// </summary>
        public Player Owner { get; set; }

        /// <summary>
        /// カードのコントローラ（現在のプレイヤー）
        /// </summary>
        public Player Player { get; set; }

        /// <summary>
        /// 現在いるエリア
        /// </summary>
        public AreaType Area { get; set; }

        /// <summary>
        /// ピクチャボックス
        /// </summary>
        public PictureBox Picture { get; private set; }

        /// <summary>
        /// アクティブ
        /// </summary>
        public bool Active { get; set; }

        /// <summary>
        /// 表裏
        /// </summary>
        public bool FaceSide { get; set; }

        /// <summary>
        /// 変身
        /// </summary>
        public bool Transformed { get; set; }

        /// <summary>
        /// 公開状態
        /// </summary>
        public bool Public { get; set; }

        /// <summary>
        /// カードの表画像
        /// </summary>
        public Image CardImage { get; set; }

        /// <summary>
        /// カード画像があるか
        /// </summary>
        public bool IsImageExist { get; set; }

        /// <summary>
        /// 場における仮想座標
        /// </summary>
        public PointF VirtualPoint { get; set; }

        /// <summary>
        /// 裏向きカード時の名前
        /// </summary>
        public CardInfo BackCardInfo { get; set; }

        /// <summary>
        /// 表示名
        /// </summary>
        public string NickName {
            get {
                if (Area == AreaType.Nodes) { return (Active ? "アクティブ" : "スリープ") + "ノード"; }
                if (Area == AreaType.Field && !FaceSide) {
                    return BackCardInfo.Name;
                }
                return Info.Name;
            }
        }

        /// <summary>
        /// 上塗りの色
        /// </summary>
        public Color CoverColor {
            get { return this.labelCover.BackColor; }
            set { this.labelCover.BackColor = value; }
        }

        private string tooltip_string { get; set; }

        #endregion


        #region 表示関係


        /// <summary>
        /// カードの読み込み
        /// </summary>
        /// <param name="card"></param>
        public void LoadCard( CardInfo card ) {
            m_card = card;
            Name = card.Name;
        }



        /// <summary>
        /// 仮想座標を使って移動
        /// </summary>
        /// <param name="point">移動したい仮想座標</param>
        public void MoveVirtualPoint( PointF point ) {
            int x = (int)( point.X * (float)Parent.Width );
            int y = (int)( point.Y * (float)Parent.Height );
            if (Player.IsUpSide) { y = Parent.Height - y; }
            x -= Size.Width / 2;
            y -= Size.Height / 2;

            // 範囲処理
            int margin = 12;
            if (x + Size.Width > Parent.Width - margin) { x = Parent.Width - Size.Width - margin; }
            if (y + Size.Height > Parent.Height - margin) { y = Parent.Height - Size.Height - margin; }
            if (x < margin) { x = margin; }
            if (y < margin) { y = margin; }

            Location = new Point(x, y);
            VirtualPoint = point;
        }

        /// <summary>
        /// 仮想座標を得る
        /// </summary>
        /// <param name="point">置きたい実座標</param>
        /// <param name="parent">親コントロールのサイズ</param>
        /// <returns>算出された仮想座標</returns>
        public PointF GetVirtualPoint(Point point, Size parent) {
            int xx = point.X; int yy = point.Y;
            xx += Size.Width / 2; yy += Size.Height / 2;
            if (Player.IsUpSide) { yy = parent.Height - yy; }
            float x = (float)xx / (float)parent.Width;
            float y = (float)yy / (float)parent.Height;
            return new PointF(x, y);
        }

        /// <summary>
        /// カードの再描画
        /// </summary>
        public void Redraw() {

            Size base_size = Properties.Settings.Default.CardSize;
            if (Active) { this.Size = base_size; }
            else { this.Size = new Size(base_size.Height, base_size.Width); }

            // 画像の選択
//            if (FaceSide && !(Area == AreaType.Hands && Player.Type != PlayerType.Control)) {
            if (FaceSide || Area == AreaType.Hands && Player.Type == PlayerType.Control) {
                this.pictureBox.BackgroundImage = getImage(Active, Transformed ^ Player.IsUpSide);
            }
            else {
                this.pictureBox.BackgroundImage = getBackImage(Active, Transformed ^ Player.IsUpSide);
            }
                
            // 文字表示
            if ((!IsImageExist || Properties.Settings.Default.AlwaysCardLabel)
                && (Area == AreaType.Field || Area == AreaType.Hands || Area == AreaType.Grave)
                && !(Area == AreaType.Hands && Player.Type != PlayerType.Control)) {
                CardInfo side = (FaceSide || Area == AreaType.Hands) ? Info : BackCardInfo;
                labelName.Font = new Font("MS UI Gothic", Properties.Settings.Default.CardFontSize);
                if (labelName.Text != side.ShortName) {
                    labelName.Text = side.ShortName;
                    labelStatus.Text = side.BattleStatus;
                    labelGraze.Text = "G" + side.Glaze;
                    labelSkill.Text = side.Skill;
                }
                labelName.Visible = true;
                labelStatus.Visible = labelStatus.Text != "-/-";
                labelGraze.Visible = labelGraze.Text != "G-";
                labelSkill.Visible = true;
                labelName.MaximumSize = Size;


            }
            else {
                labelName.Visible = false;
                labelStatus.Visible = false;
                labelGraze.Visible = false;
                labelSkill.Visible = false;

            }


            Refresh();
        }

        private Image[] imgCache = new Image[4];
        /// <summary>
        /// 最適なイメージを選択
        /// </summary>
        /// <param name="active">アクティブ状態</param>
        /// <param name="reverse">リバース状態</param>
        /// <returns>最適なイメージ</returns>
        private Image getImage(bool active, bool reverse) {
            if (CardImage == null) {
                string path = Application.StartupPath + Properties.Settings.Default.ImagePath + Info.FileName;
                if (File.Exists(path)) { CardImage = Image.FromFile(path); IsImageExist = true; }
                else { CardImage = CardBox.GenericImage; IsImageExist = false; }
            }
            int index = 0;
            if (!active) { index += 1; }
            if (reverse) { index += 2; }
            if (imgCache[index] == null) {
                RotateFlipType[] flips = { RotateFlipType.RotateNoneFlipNone, RotateFlipType.Rotate90FlipNone, RotateFlipType.Rotate180FlipNone, RotateFlipType.Rotate270FlipNone };
                imgCache[index] = new Bitmap(CardImage);
                imgCache[index].RotateFlip(flips[index]);
            }

            return imgCache[index];
        }
        /// <summary>
        /// 最適な裏画像を選択
        /// </summary>
        /// <param name="active">アクティブ状態</param>
        /// <param name="reverse">リバース状態</param>
        /// <returns>最適なイメージ</returns>
        private Image getBackImage( bool active, bool reverse ) {
            int index = 0;
            if (!active) { index += 1; }
            if (reverse) { index += 2; }
            if (CardBox.CachedBackImages[index] == null) {
                RotateFlipType[] flips = { RotateFlipType.RotateNoneFlipNone, RotateFlipType.Rotate90FlipNone, RotateFlipType.Rotate180FlipNone, RotateFlipType.Rotate270FlipNone };
                CardBox.CachedBackImages[index] = new Bitmap(CardBox.BackImage);
                CardBox.CachedBackImages[index].RotateFlip(flips[index]);
            }

            return CardBox.CachedBackImages[index];
        }

        #endregion

        private void CardBox_Load( object sender, EventArgs e ) {
        }




















 




    }
}
