#include "pch.hpp"
#include "dlgcommon.hpp"
#include "rankprofdlg.hpp"
#include "trfilterdlg.hpp"
#include "trackrecorddlg.hpp"
#include "scoreline.hpp"
#include "resource.h"

#define UC_REFLESH 0xEE00

#define EVEN_COLOR 0xFFFFFF
#define ODD_COLOR  0xD0DCD0

#define PROP_SORT_DIRECTION   _T("SORT.DIRECTION")
#define PROP_SORT_CURRENTCOL  _T("SORT.CURRENTCOL")

enum LVSORTDIRECTION {
	LVSORTDIR_ASCEND =   1,
	LVSORTDIR_DESCEND = -1
};

enum LVSORTTYPE {
  LVSORT_STRING,
  LVSORT_INTEGER
};

struct LVSORTDESC {
	HWND hwndListView;
	int currentCol;
	LVSORTDIRECTION direction;
};

static struct {
	DWORD fmt;
	int cx;
	LPCTSTR text;
	LVSORTTYPE sorttype;
} s_listColumns[] = {
	{ LVCFMT_LEFT,  100, _T(""),	LVSORT_STRING },
	{ LVCFMT_RIGHT,  55, _T("ΐ퐔"),	LVSORT_INTEGER },
	{ LVCFMT_RIGHT,  55, _T(""),		LVSORT_INTEGER },
	{ LVCFMT_RIGHT,  55, _T(""),		LVSORT_INTEGER },
	{ LVCFMT_RIGHT,  55, _T(""),	LVSORT_INTEGER }
};

static void RankProfDialog_QueryCallback(SCORELINE_ITEM *item, void *user)
{
	HWND listWnd = (HWND)user;

	TCHAR text[256];
	LVITEM lvitem;
	lvitem.mask = LVIF_TEXT;
	lvitem.cchTextMax = sizeof item->p2name;
	lvitem.iItem = 0;
	lvitem.iSubItem = 0;
	lvitem.pszText = item->p2name;
	ListView_InsertItem(listWnd, &lvitem);

	lvitem.iSubItem = 1;
	lvitem.pszText = text;
	lvitem.cchTextMax = sizeof text;
	::wsprintf(text, _T("%d"), item->p1win + item->p2win);
	ListView_SetItem(listWnd, &lvitem);

	lvitem.iSubItem = 2;
	::wsprintf(text, _T("%d"), item->p1win);
	ListView_SetItem(listWnd, &lvitem);

	lvitem.iSubItem = 3;
	::wsprintf(text, _T("%d"), item->p2win);
	ListView_SetItem(listWnd, &lvitem);

	int sum = item->p1win + item->p2win;
	lvitem.iSubItem = 4;
	::wsprintf(text, _T("%d"), ::MulDiv(item->p1win, 100, sum ? sum : 1));
	ListView_SetItem(listWnd, &lvitem);
}

static void SysMenu_OnClose(HWND hDlg, int x, int y)
{
	DestroyWindow(hDlg);
}

static void SysMenu_OnReflesh(HWND hDlg, int x, int y)
{
	HWND listWnd = GetDlgItem(hDlg, IDC_LIST_SCORELINE);
	ListView_DeleteAllItems(listWnd);
	SCORELINE_FILTER_DESC filterDesc;
	filterDesc.mask = 0;
	ScoreLine_QueryProfileRank(filterDesc, RankProfDialog_QueryCallback, (void*)listWnd);
}

int CALLBACK RankProfView_SortAsString(LPARAM lParam1, LPARAM lParam2, LPARAM lParamSort)
{
	LVSORTDESC& sortdesc = *reinterpret_cast<LVSORTDESC*>(lParamSort);

	TCHAR textA[256], textB[256];
	ListView_GetItemText(sortdesc.hwndListView, lParam1, sortdesc.currentCol, textA, sizeof textA - 1);
	ListView_GetItemText(sortdesc.hwndListView, lParam2, sortdesc.currentCol, textB, sizeof textB - 1);
	return ::lstrcmp(textA, textB) * sortdesc.direction;
}

int CALLBACK RankProfView_SortAsInteger(LPARAM lParam1, LPARAM lParam2, LPARAM lParamSort)
{
	LVSORTDESC& sortdesc = *reinterpret_cast<LVSORTDESC*>(lParamSort);

	TCHAR textA[256], textB[256];
	ListView_GetItemText(sortdesc.hwndListView, lParam1, sortdesc.currentCol, textA, sizeof textA - 1);
	ListView_GetItemText(sortdesc.hwndListView, lParam2, sortdesc.currentCol, textB, sizeof textB - 1);

	int valA, valB;
	valA = ::StrToInt(textA);
	valB = ::StrToInt(textB);

	return valA < valB ? -sortdesc.direction : valA > valB ? sortdesc.direction : 0;
}

static LRESULT RankProfView_OnColumnClick(HWND hwndParent, LPNMLISTVIEW lpnmListView)
{
	LVSORTDESC sortdesc = {
		lpnmListView->hdr.hwndFrom,
		reinterpret_cast<int>(::GetProp(hwndParent, PROP_SORT_CURRENTCOL)),
		static_cast<LVSORTDIRECTION>(reinterpret_cast<int>(::GetProp(hwndParent, PROP_SORT_DIRECTION)))
	};

	if(lpnmListView->iSubItem != sortdesc.currentCol) {
		sortdesc.currentCol = lpnmListView->iSubItem;
		sortdesc.direction  = LVSORTDIR_ASCEND;
		::SetProp(hwndParent, PROP_SORT_CURRENTCOL, reinterpret_cast<HANDLE>(sortdesc.currentCol));
		::SetProp(hwndParent, PROP_SORT_DIRECTION, reinterpret_cast<HANDLE>(sortdesc.direction));
	} else {
		sortdesc.direction = sortdesc.direction == LVSORTDIR_ASCEND ? LVSORTDIR_DESCEND : LVSORTDIR_ASCEND;
		::SetProp(hwndParent, PROP_SORT_DIRECTION, reinterpret_cast<HANDLE>(sortdesc.direction));
	}

	switch(s_listColumns[sortdesc.currentCol].sorttype) {
	case LVSORT_STRING:
		ListView_SortItemsEx(sortdesc.hwndListView, RankProfView_SortAsString, reinterpret_cast<LPARAM>(&sortdesc));
		break;
	case LVSORT_INTEGER:
		ListView_SortItemsEx(sortdesc.hwndListView, RankProfView_SortAsInteger, reinterpret_cast<LPARAM>(&sortdesc));
		break;
	}

	return TRUE;
}

static LRESULT RankProfView_OnCustomDraw(HWND hwndParent, LPNMLVCUSTOMDRAW lpnlvCustomDraw)
{
	switch(lpnlvCustomDraw->nmcd.dwDrawStage) {
	case CDDS_PREPAINT:
		return CDRF_NOTIFYITEMDRAW;
	case CDDS_ITEMPREPAINT:
		lpnlvCustomDraw->clrTextBk = 
			(lpnlvCustomDraw->nmcd.dwItemSpec & 1) ?
				EVEN_COLOR : ODD_COLOR;
		return CDRF_NOTIFYSUBITEMDRAW;
	default:
		return CDRF_DODEFAULT;
	}
}

static LRESULT RankProfView_OnDoubleClick(HWND hwndParent, HWND hwnd)
{
	int index = ListView_GetSelectionMark(hwnd);
	if (index >= 0) {
		LVITEM item;
		item.mask = LVIF_TEXT;
		item.iItem = index;
		item.iSubItem = 0;
		item.cchTextMax = sizeof g_trfltP2Name;
		item.pszText = g_trfltP2Name;
		ListView_GetItem(hwnd, &item);

		g_trfltP2NameEnabled = true;

		TrackRecordDialog_ShowModeless(GetParent(hwndParent), NULL);
	}
	return FALSE;
}

static void RankProfDialog_OnSysCommand(HWND hDlg, UINT nID, int x, int y)
{
	if(nID == SC_CLOSE) {
		SysMenu_OnClose(hDlg, x, y);
	} else if(nID == UC_REFLESH) {
		SysMenu_OnReflesh(hDlg, x, y);
	}
}

static LRESULT RankProfDialog_OnNotify(HWND hDlg, int idCtrl, LPNMHDR pNMHdr)
{
	switch(idCtrl) {
	case IDC_LIST_SCORELINE:
		switch(pNMHdr->code) {
		case NM_CUSTOMDRAW:
			return RankProfView_OnCustomDraw(hDlg, reinterpret_cast<LPNMLVCUSTOMDRAW>(pNMHdr));
		case NM_DBLCLK:
			return RankProfView_OnDoubleClick(hDlg, pNMHdr->hwndFrom);
		case LVN_COLUMNCLICK:
			return RankProfView_OnColumnClick(hDlg, reinterpret_cast<LPNMLISTVIEW>(pNMHdr));
		}
		break;
	}
	return FALSE;
}

static BOOL RankProfDialog_InitSysMenu(HWND hDlg)
{
	HMENU hSysMenu = ::GetSystemMenu(hDlg, FALSE);
	int itemIndex = 0;
	::InsertMenu(hSysMenu, itemIndex++, MF_STRING | MF_BYPOSITION, UC_REFLESH, _T("ŐV̏ɍXV"));
	::InsertMenu(hSysMenu, itemIndex++, MF_SEPARATOR | MF_BYPOSITION, 0, NULL);
	return TRUE;
}

static BOOL RankProfDialog_InitListView(HWND hDlg)
{
	::SetProp(hDlg, PROP_SORT_CURRENTCOL, reinterpret_cast<HANDLE>(0));
	::SetProp(hDlg, PROP_SORT_DIRECTION, reinterpret_cast<HANDLE>(LVSORTDIR_ASCEND));

	HWND hwndListView = GetDlgItem(hDlg, IDC_LIST_SCORELINE);
	ListView_SetExtendedListViewStyle(hwndListView, LVS_EX_GRIDLINES | LVS_EX_FULLROWSELECT);

	LVCOLUMN column;
	column.mask	= LVCF_FMT | LVCF_TEXT | LVCF_WIDTH;
	for(int i = 0; i < ARRAYSIZE(s_listColumns); ++i) {
		column.fmt = s_listColumns[i].fmt;
		column.cx = s_listColumns[i].cx;
		column.cchTextMax = ::lstrlen(s_listColumns[i].text);
		column.pszText = const_cast<LPTSTR>(s_listColumns[i].text);
		ListView_InsertColumn(hwndListView, i, &column);
	}
	SCORELINE_FILTER_DESC filterDesc;
	filterDesc.mask = 0;
	ScoreLine_QueryProfileRank(filterDesc, RankProfDialog_QueryCallback, reinterpret_cast<void*>(hwndListView));

	LVSORTDESC sortdesc = { hwndListView, 0, LVSORTDIR_ASCEND };
	switch(s_listColumns[0].sorttype) {
	case LVSORT_STRING:
		ListView_SortItemsEx(hwndListView, RankProfView_SortAsString, reinterpret_cast<LPARAM>(&sortdesc));
		break;
	case LVSORT_INTEGER:
		ListView_SortItemsEx(hwndListView, RankProfView_SortAsInteger, reinterpret_cast<LPARAM>(&sortdesc));
		break;
	}
	return TRUE;
}

static BOOL RankProfDialog_OnInitDialog(HWND hDlg, HWND hwndFocus, LPARAM lParam)
{
	SetWindowText(hDlg, "vt@C");
	RankProfDialog_InitSysMenu(hDlg);
	RankProfDialog_InitListView(hDlg);
	return TRUE;
}

BOOL CALLBACK RankProfDialog_DlgProc(HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam)
{
	switch(Msg) {
	HANDLE_DLG_MSG(hDlg, WM_INITDIALOG, RankProfDialog_OnInitDialog);
	HANDLE_DLG_MSG(hDlg, WM_SYSCOMMAND, RankProfDialog_OnSysCommand);
	HANDLE_DLG_MSG(hDlg, WM_NOTIFY, RankProfDialog_OnNotify);
	}
	return FALSE;
}
