/*
 * Redistribution and use in source and binary forms, with
 * or without modification, are permitted provided that the
 * following conditions are met:
 *
 * 1. Redistributions of source code must retain this list
 *    of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce this
 *    list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#include <config.h>
#include <stdlib.h>
#include <string.h>

#include "datpacker.h"
#include "thdat.h"

/* TODO: Something smarter here, and in the options... */
extern archive_module_t archive_th02;
extern archive_module_t archive_th06;
extern archive_module_t archive_th75;
extern archive_module_t archive_th08;
extern archive_module_t archive_th95;

char library_error[LIBRARY_ERROR_SIZE] = { 0 };

static void
print_usage()
{
	puts("Usage: datpacker [OPTION]... ARCHIVE FILE...\n"
	     "OPTION can be:\n"
	     "  -h           prints the usage\n"
	     "  -v VERSION   can be 2, 6, 7, 75, 8, 9, 95, 10, 11, or 12, defaults to 12\n"
	     "  -V           print program version\n\n"
	     "Report bugs to <" PACKAGE_BUGREPORT ">.");
}

static void
print_version()
{
	puts("This is " PACKAGE_NAME " release " PACKAGE_VERSION ".");
}

static void
archive_free(archive_t* archive)
{
	if (archive) {
		free(archive->entries);
		free(archive);
	}
}

int
main(int argc, char* argv[])
{
	FILE* archive;
	archive_t* private;
	archive_module_t* archive_module = NULL;
	unsigned int version = 12;
	int i;
	int argstart;

	for (i = 1; i < argc; ++i) {
		if (strcmp(argv[i], "-h") == 0) {
			print_usage();
			return EXIT_SUCCESS;
		} else if (strcmp(argv[i], "-V") == 0) {
			print_version();
			return EXIT_SUCCESS;
		} else if (strcmp(argv[i], "-v") == 0) {
			++i;
			if (i == argc) {
				print_usage();
				return EXIT_FAILURE;
			}
			version = strtol(argv[i], NULL, 10);
		} else {
			break;
		}
	}

	if ((i + 2) > argc) {
		print_usage();
		return EXIT_FAILURE;
	}

	switch (version) {
	case 2:
		archive_module = &archive_th02;
		break;
	case 6:
	case 7:
		archive_module = &archive_th06;
		break;
	case 75:
		archive_module = &archive_th75;
		break;
	case 8:
	case 9:
		archive_module = &archive_th08;
		break;
	case 95:
	case 10:
	case 11:
	case 12:
		archive_module = &archive_th95;
		break;
	}

	if (!archive_module) {
		fprintf(stderr, "datpacker: unsupported version\n");
		return EXIT_FAILURE;
	}

	archive = fopen(argv[i], "wb");
	if (!archive) {
		fprintf(stderr, "datpacker: couldn't open %s for writing\n", argv[i]);
		return EXIT_FAILURE;
	}

	/* XXX: Is this count correct? */
	private = archive_module->open(archive, version, argc - i - 1);
	if (!private) {
		fprintf(stderr, "datpacker: %s\n", library_error);
		fclose(archive);
		return EXIT_FAILURE;
	}

	argstart = i + 1;
#pragma omp parallel shared(private)
#pragma omp for
	for (i = argstart; i < argc; ++i) {
		entry_t* entry;
		FILE* fd = fopen(argv[i], "rb");
		if (!fd) {
			fprintf(stderr, "datpacker: couldn't open %s for reading\n", argv[i]);
			fclose(archive);
			exit(EXIT_FAILURE);
		}
		printf("%s...\n", argv[i]);
		fflush(stdout);

		/* TODO: Move the definition to this file? */
		entry = archive_add_entry(private, fd, argv[i], archive_module->flags);
		if (!entry || archive_module->write(private, entry, fd) == -1) {
			fprintf(stderr, "datpacker: %s\n", library_error);
			fclose(fd);
			fclose(archive);
			archive_free(private);
			exit(EXIT_FAILURE);
		}

		fclose(fd);
	}

	if (archive_module->close(private) == -1) {
		fprintf(stderr, "datpacker: %s\n", library_error);
		fclose(archive);
		archive_free(private);
		return EXIT_FAILURE;
	}

	archive_free(private);
	fclose(archive);

	return EXIT_SUCCESS;
}
