/*
 * Redistribution and use in source and binary forms, with
 * or without modification, are permitted provided that the
 * following conditions are met:
 *
 * 1. Redistributions of source code must retain this list
 *    of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce this
 *    list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#include <config.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

#include "datpacker.h"
#include "thdat.h"
#include "util.h"

static archive_t*
th02_open(FILE* fd, unsigned int version, unsigned int count)
{
	return archive_open(fd, version, (count + 1) * 32, count);
}

/* TODO: Check that name is 8.3... maybe a THDAT_ flag? */
/* XXX: What about lowercase names... are they supported? */
static int
th02_write(archive_t* archive, entry_t* entry, FILE* fd)
{
	unsigned int i;
	unsigned char* data;

	for (i = 0; i < 13; ++i)
		if (entry->name[i])
			entry->name[i] ^= 0xff;

	data = thdat_read_file(entry, fd);
	if (!data)
		return -1;

	data = thdat_rle(entry, data);
	if (!data)
		return -1;

	for (i = 0; i < entry->zsize; ++i)
		data[i] ^= 0x12;

	return thdat_write_entry(archive, entry, data);
}

static int
th02_close(archive_t* archive)
{
	unsigned char* buffer;
	unsigned char* buffer_ptr;
	unsigned int i;
	unsigned int list_size = 32 * (archive->count + 1);

	if (thdat_seek(archive->fd, 0) == -1)
		return -1;

	buffer = malloc(list_size);
	if (!buffer) {
		snprintf(library_error, LIBRARY_ERROR_SIZE, "malloc failed: %s", strerror(errno));
		return -1;
	}
	memset(buffer, 0, list_size);

	buffer_ptr = buffer;
	for (i = 0; i < archive->count; ++i) {
		entry_t* entry = &archive->entries[i];

		if (entry->zsize == entry->size)
			buffer_ptr = write_uint16(0xf388, buffer_ptr);
		else
			buffer_ptr = write_uint16(0x9595, buffer_ptr);

		*buffer_ptr++ = 3;

		buffer_ptr = write_bytes((unsigned char*)entry->name, 13, buffer_ptr);
		buffer_ptr = write_uint32(entry->zsize, buffer_ptr);
		buffer_ptr = write_uint32(entry->size, buffer_ptr);
		buffer_ptr = write_uint32(entry->offset, buffer_ptr);
		buffer_ptr += 4;
	}

	if (fwrite(buffer, list_size, 1, archive->fd) != 1) {
		snprintf(library_error, LIBRARY_ERROR_SIZE, "couldn't write: %s", strerror(errno));
		free(buffer);
		return -1;
	}
	free(buffer);

	return 0;
}

const archive_module_t archive_th02 = {
	th02_open,
	th02_write,
	th02_close,
	THDAT_BASENAME
};
