/*
 * Redistribution and use in source and binary forms, with
 * or without modification, are permitted provided that the
 * following conditions are met:
 *
 * 1. Redistributions of source code must retain this list
 *    of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce this
 *    list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
#include <config.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#ifdef HAVE_LIBGEN_H
#include <libgen.h>
#endif

#include "datpacker.h"
#include "util.h"
#include "thdat.h"
#include "thrle.h"
#include "thlzss.h"

archive_t*
archive_open(FILE* fd, uint32_t version, uint32_t offset, unsigned int count)
{
	archive_t* archive;

	/* Write the header later. */
	if (fseek(fd, offset, SEEK_SET) == -1) {
		snprintf(library_error, LIBRARY_ERROR_SIZE, "couldn't seek: %s", strerror(errno));
		return NULL;
	}

	archive = malloc(sizeof(archive_t));
	if (!archive) {
		snprintf(library_error, LIBRARY_ERROR_SIZE, "malloc failed: %s", strerror(errno));
		return NULL;
	}

	archive->entries = malloc(count * sizeof(entry_t));
	if (!archive->entries) {
		snprintf(library_error, LIBRARY_ERROR_SIZE, "malloc failed: %s", strerror(errno));
		free(archive);
		return NULL;
	}

	archive->version = version;
	archive->fd = fd;
	archive->offset = offset;
	archive->count = 0;

	return archive;
}

entry_t*
archive_add_entry(archive_t* archive, FILE* fd, const char* filename, unsigned int flags)
{
	int i;
	char* path;
	char* name;
	entry_t* e;

#pragma omp critical
	e = &archive->entries[archive->count++];

	memset(e->name, 0, 256);
	e->size = fsize(fd);
	e->zsize = 0;
	e->offset = 0;
	e->extra = 0;

	path = strdup(filename);
	if (flags & THDAT_BASENAME) {
#ifdef WIN32
		char filename[_MAX_FNAME];
		char ext[_MAX_EXT];
		_splitpath(path, NULL, NULL, filename, ext);
		_snprintf(path, strlen(path), "%s%s", filename, ext);
		name = path;
#else
		name = basename(path);
#endif
	} else {
		name = path;
	}


	strncpy(e->name, name, 255);

	/* TODO: Process this after adding all files instead. */
#pragma omp critical
	for (i = 0; i < archive->count - 1; ++i) {
		if (&archive->entries[i] != e && strcmp(name, archive->entries[i].name) == 0) {
			snprintf(library_error, LIBRARY_ERROR_SIZE, "duplicate filename ``%s''", name);
			i = -1;
			break;
		}
	}

	free(path);

	if (i == -1)
		return NULL;

	if (e->size == -1) {
		snprintf(library_error, LIBRARY_ERROR_SIZE, "fsize failed: %s", strerror(errno));
		return NULL;
	}

	return e;
}

unsigned char*
thdat_read_file(entry_t* entry, FILE* fd)
{
	unsigned char* data = malloc(entry->size);
	if (!data) {
		snprintf(library_error, LIBRARY_ERROR_SIZE, "malloc failed: %s", strerror(errno));
		return NULL;
	}

	if (fread(data, entry->size, 1, fd) != 1) {
		snprintf(library_error, LIBRARY_ERROR_SIZE, "couldn't read: %s", strerror(errno));
		free(data);
		return NULL;
	}

	return data;
}

unsigned char*
thdat_read_file_lzss(entry_t* entry, FILE* fd)
{
	unsigned char* data = th_lz_fd(fd, &entry->zsize);
	if (!data) {
		snprintf(library_error, LIBRARY_ERROR_SIZE, "th_lz_fd failed (TODO: Improve error message.)");
		return NULL;
	}

	return data;
}

unsigned char*
thdat_rle(entry_t* entry, unsigned char* data)
{
	unsigned char* zdata = th_rle(data, entry->size, &entry->zsize);
	if (!zdata) {
		snprintf(library_error, LIBRARY_ERROR_SIZE, "th_rle failed (TODO: Improve error message.)");
		return NULL;
	}

	if (entry->zsize >= entry->size) {
		entry->zsize = entry->size;
		free(zdata);
		zdata = data;
	} else {
		free(data);
	}

	return zdata;
}

int
thdat_write_entry(archive_t* archive, entry_t* entry, unsigned char* data)
{
	size_t ret;

#pragma omp critical
{
	ret = fwrite(data, entry->zsize, 1, archive->fd);
	entry->offset = archive->offset;
	archive->offset += entry->zsize;
}

	free(data);

	if (ret != 1) {
		snprintf(library_error, LIBRARY_ERROR_SIZE, "couldn't write: %s", strerror(errno));
		free(data);
		return -1;
	} else
		return 0;
}

int
thdat_seek(FILE* fd, long offset)
{
	if (fseek(fd, offset, SEEK_SET) == -1) {
		snprintf(library_error, LIBRARY_ERROR_SIZE, "couldn't seek: %s", strerror(errno));
		return -1;
	}

	return 0;
}

static int
entry_compar(const void* a, const void* b)
{
	entry_t* ea = (entry_t*)a;
	entry_t* eb = (entry_t*)b;
	if (ea->offset < eb->offset)
		return -1;
	else if (ea->offset > eb->offset)
		return 1;
	else
		return 0;
}

void
thdat_sort(archive_t* archive)
{
	qsort(archive->entries, archive->count, sizeof(entry_t), entry_compar);
}
