#include "pch.hpp"
#include "scoreline.hpp"
#include "MappedFile.hpp"
#include "Formatter.hpp"
#include "SWRSAddrDef.h"

#include <sqlite3.h>
#include <utility>

#include "MinimalMemory.hpp"

#define MINIMAL_USE_PROCESSHEAPSTRING
#include "MinimalPath.hpp"

static sqlite3 *s_db;
static bool s_viewEntered;

static DWORD s_scoreLine[SWRSCHAR_MAX][SWRSCHAR_MAX][2];
static DWORD s_scoreLineNew[SWRSCHAR_MAX][SWRSCHAR_MAX][2];
static Minimal::ProcessHeapStringT<char> s_srPath;
static Minimal::ProcessHeapStringT<char> s_srPathU;

void ScoreLine_SetPath(LPCSTR path)
{
	s_srPath = path;
	Minimal::ToUTF8(s_srPathU, path);
}

LPCSTR ScoreLine_GetPath()
{
	return s_srPath;
}

static int ScoreLine_QueryCallback(void *, int argc, char **argv, char **colName)
{
	if(argc != 4) return 1;

	int p1id = StrToInt(argv[0]);
	int p2id = StrToInt(argv[1]);
	int p1win = StrToInt(argv[2]);
	int p2win = StrToInt(argv[3]);

	if(p1id < 0 || p1id >= SWRSCHAR_MAX) return 1;
	if(p2id < 0 || p2id >= SWRSCHAR_MAX) return 1;
	if(p1win < 0) p1win = 0;
	if(p2win < 0) p2win = 0;

	s_scoreLineNew[p1id][p2id][0] = p1win;
	s_scoreLineNew[p1id][p2id][1] = p2win;
	return 0;
}

static int ScoreLine_QueryCallback2(void *user, int argc, char **argv, char **colName)
{
	if(argc != 3) return 1;

	SCORELINE_ITEM item;
	::lstrcpy(item.p2name, argv[0]);
	item.p1win = StrToInt(argv[1]);
	item.p2win = StrToInt(argv[2]);

	std::pair<void(*)(SCORELINE_ITEM *, void *), void*> *cbinfo;
	*(void**)&cbinfo = user;

	cbinfo->first(&item, cbinfo->second);

	return 0;
}

static int ScoreLine_QueryCallback3(void *user, int argc, char **argv, char **colName)
{
	if(argc != 7) return 1;

	SCORELINE_ITEM item;
	StrToInt64Ex(argv[0], STIF_DEFAULT, &item.timestamp);
	::lstrcpy(item.p1name, argv[1]);
	item.p1id  = StrToInt(argv[2]);
	item.p1win = StrToInt(argv[3]);

	::lstrcpy(item.p2name, argv[4]);
	item.p2id  = StrToInt(argv[5]);
	item.p2win = StrToInt(argv[6]);

	std::pair<void(*)(SCORELINE_ITEM *, void *), void*> *cbinfo;
	*(void**)&cbinfo = user;

	cbinfo->first(&item, cbinfo->second);

	return 0;
}

static void ScoreLine_ConstructFilter(Minimal::ProcessHeapStringT<char> &ret, SCORELINE_FILTER *filters, int count)
{
	if(count == 0) return;
	char buff[64];
	ret += " WHERE ";
	for(int i = 0; i < count; ++i) {
		if(i > 0) ret += " AND ";
		switch(filters[i].type) {
		case SCORELINE_FILTER_P1NAME:
			sqlite3_snprintf(_countof(buff), buff, "p1name = '%q'", filters[i].name);
			ret += buff;
			break;
		case SCORELINE_FILTER_P2NAME:
			sqlite3_snprintf(_countof(buff), buff, "p2name = '%q'", filters[i].name);
			ret += buff;
			break;
		case SCORELINE_FILTER_P1ID:
			sqlite3_snprintf(_countof(buff), buff, "p1id = %ld", filters[i].id);
			ret += buff;
			break;
		case SCORELINE_FILTER_P2ID:
			sqlite3_snprintf(_countof(buff), buff, "p2id = %ld", filters[i].id);
			ret += buff;
			break;
		case SCORELINE_FILTER_TIMESTAMP_BEGIN:
			sqlite3_snprintf(_countof(buff), buff, "timestamp >= %lld", filters[i].timestamp);
			ret += buff;
			break;
		case SCORELINE_FILTER_TIMESTAMP_END:
			sqlite3_snprintf(_countof(buff), buff, "timestamp <= %lld", filters[i].timestamp);
			ret += buff;
			break;
		}
	}
}

bool ScoreLine_QueryTrackRecord(SCORELINE_FILTER *filters, int count)
{

	Minimal::ProcessHeapStringT<char> filterStr;
	char *errmsg;
	int rc;

	ZeroMemory(s_scoreLineNew, sizeof(s_scoreLineNew));
	ScoreLine_ConstructFilter(filterStr, filters, count);

	char *query = sqlite3_mprintf(
		"SELECT T.p1id, T.p2id, sum(T.p1win/2), sum(T.p2win/2) "
		"FROM trackrecord123 AS T %s GROUP BY T.p1id, T.p2id;",
		filterStr.GetRaw());
	rc = sqlite3_exec(s_db, query, ScoreLine_QueryCallback, NULL, &errmsg);
	sqlite3_free(query);
	if(rc) return false;

	memcpy(s_scoreLine, s_scoreLineNew, sizeof(s_scoreLine));
	return true;
}

bool ScoreLine_QueryTrackRecord(SCORELINE_FILTER *filters, int count, int limit)
{

	Minimal::ProcessHeapStringT<char> filterStr;
	char *errmsg;
	int rc;

	ScoreLine_ConstructFilter(filterStr, filters, count);

	ZeroMemory(s_scoreLineNew, sizeof(s_scoreLineNew));
	char *query = sqlite3_mprintf(
		"SELECT p1id, p2id, sum(p1win/2), sum(p2win/2) "
		"FROM ("
			"SELECT p1id, p2id, p1win, p2win "
			"FROM trackrecord123 %s ORDER BY timestamp DESC LIMIT %d"
		")"
		"GROUP BY p1id, p2id;",
		filterStr.GetRaw(), limit);
	rc = sqlite3_exec(s_db, 
		query, ScoreLine_QueryCallback, NULL, &errmsg);
	sqlite3_free(query);
	if(rc) return false;

	memcpy(s_scoreLine, s_scoreLineNew, sizeof(s_scoreLine));
	return true;
}

DWORD ScoreLine_Read(int p1, int p2, int idx)
{
	return s_scoreLine[p1][p2][idx];
}

bool ScoreLine_Append(SCORELINE_ITEM *item)
{
	sqlite3_stmt *stmt;
	int rc;

	if(!s_db) return false;

	rc = sqlite3_prepare(s_db, 
		"insert into trackrecord123 (timestamp, p1name, p1id, p1win, p2name, p2id, p2win) "
		 "values (?, ?, ?, ?, ?, ?, ?);", -1, &stmt, NULL);
	if(rc) return false;

	sqlite3_bind_int64(stmt, 1, item->timestamp);
	sqlite3_bind_text (stmt, 2, item->p1name, -1, NULL);
	sqlite3_bind_int  (stmt, 3, item->p1id);
	sqlite3_bind_int  (stmt, 4, item->p1win);
	sqlite3_bind_text (stmt, 5, item->p2name, -1, NULL);
	sqlite3_bind_int  (stmt, 6, item->p2id);
	sqlite3_bind_int  (stmt, 7, item->p2win);
	bool ret = (sqlite3_step(stmt) == SQLITE_DONE);

	sqlite3_finalize(stmt);

	return ret;
}

bool ScoreLine_Remove(time_t timestamp)
{
	if(!s_db) return false;
	char *query = sqlite3_mprintf(
		"DELETE FROM trackrecord123 WHERE timestamp = %lld",
		timestamp);
	int rc = sqlite3_exec(s_db, query, NULL, NULL, NULL);
	sqlite3_free(query);
	if(rc) return false;

	return true;
}

bool ScoreLine_QueryProfileRank(SCORELINE_FILTER *filters, int count, void(*callback)(SCORELINE_ITEM *, void *), void *user)
{
	if(!s_db) return false;

	Minimal::ProcessHeapStringT<char> filterStr;
	ScoreLine_ConstructFilter(filterStr, filters, count);

	std::pair<void(*)(SCORELINE_ITEM *, void *), void*> cbinfo
		= std::make_pair(callback, user);
	char *query;
	query = sqlite3_mprintf(
		"SELECT T.p2name, sum(T.p1win/2), sum(T.p2win/2) "
		"FROM trackrecord123 AS T %s GROUP BY p2name",
		filterStr.GetRaw());
	int rc = sqlite3_exec(s_db, query, ScoreLine_QueryCallback2, (void*)&cbinfo, NULL);
	sqlite3_free(query);
	if(rc) return false;

	return true;
}

bool ScoreLine_QueryTrackRecordLog(SCORELINE_FILTER *filters, int count, int limit, void(*callback)(SCORELINE_ITEM *, void *), void *user)
{
	if(!s_db) return false;

	Minimal::ProcessHeapStringT<char> filterStr;
	ScoreLine_ConstructFilter(filterStr, filters, count);

	std::pair<void(*)(SCORELINE_ITEM *, void *), void*> cbinfo
		= std::make_pair(callback, user);

	Minimal::ProcessHeapStringT<char> queryStr;

	char *query;
	if(limit > 0) {
		query = sqlite3_mprintf(
			"SELECT * FROM trackrecord123 %s ORDER BY timestamp DESC LIMIT %d",
			filterStr.GetRaw(), limit);
	} else {
		query = sqlite3_mprintf(
			"SELECT * FROM trackrecord123 %s ORDER BY timestamp DESC",
			filterStr.GetRaw());
	}
	int rc = sqlite3_exec(s_db, query, ScoreLine_QueryCallback3, (void*)&cbinfo, NULL);
	sqlite3_free(query);
	if(rc) return false;

	return true;

}

bool ScoreLine_Open(bool create)
{
	sqlite3 *db;
	if(PathFileExists(s_srPath)) {
		// I[v
		int rc = sqlite3_open(s_srPathU, &db);
		if(rc) {
			sqlite3_close(db);
			return false;
		}
		char *errmsg;
		rc = sqlite3_exec(db,
			"PRAGMA temp_store = MEMORY;"
			"PRAGMA cache_size = 1048576;"
			, NULL, NULL, &errmsg);
		if(rc) {
			sqlite3_close(db);
			return false;
		}
	} else {
		// 쐬
		int rc = sqlite3_open(s_srPathU, &db);
		if(rc) {
			sqlite3_close(db);
			return false;
		}
		char *errmsg;
		rc = sqlite3_exec(db, 
			"CREATE TABLE trackrecord123(\n"
				"timestamp	INTEGER NOT NULL,\n"
				"p1name     STRING,\n"
				"p1id       INTEGER NOT NULL,\n"
				"p1win      INTEGER NOT NULL,\n"
				"p2name     STRING,\n"
				"p2id       INTEGER NOT NULL,\n"
				"p2win      INTEGER NOT NULL,\n"
				"PRIMARY KEY (timestamp)\n"
			")", NULL, NULL, &errmsg);
		if(rc) {
			sqlite3_close(db);
			DeleteFile(s_srPath);
			return false;
		}
		rc = sqlite3_exec(db, 
			"CREATE INDEX trackrecord123_index\n"
			"on trackrecord123 (timestamp)", NULL, NULL, &errmsg);
		if(rc) {
			sqlite3_close(db);
			DeleteFile(s_srPath);
			return false;
		}
		rc = sqlite3_exec(db,
			"PRAGMA temp_store = MEMORY;"
			"PRAGMA cache_size = 1048576;"
			, NULL, NULL, &errmsg);
		if(rc) {
			sqlite3_close(db);
			return false;
		}
	}

	ScoreLine_Close();
	s_db = db;

	return true;
}

void ScoreLine_Enter()
{
	if(!s_db) return;
	sqlite3_exec(s_db, "BEGIN", NULL, NULL, NULL);
}

void ScoreLine_Leave(bool failed)
{
	if(!s_db) return;
	sqlite3_exec(s_db, failed ? "ROLLBACK" : "COMMIT", NULL, NULL, NULL);
}

void ScoreLine_Close()
{
	if(s_db) {
		sqlite3_close(s_db);
		s_db = NULL;
	}
}
