#pragma once

#include "MinimalAllocator.hpp"

namespace Minimal {

namespace MinimalString {

__inline
size_t MinimalStringLength(const wchar_t *str)
{
	return ::lstrlenW(str);
}

__inline
size_t MinimalStringLength(const char *str)
{
	return ::lstrlenA(str);
}

__inline
wchar_t * MinimalStringCopy(wchar_t *dst, const wchar_t *src)
{
	return ::lstrcpyW(dst, src);
}

__inline
char * MinimalStringCopy(char *dst, const char *src)
{
	return ::lstrcpyA(dst, src);
}

}

template<typename CharType>
class MinimalStringT
{
public:
	typedef CharType * StringType;
	typedef const CharType * ConstStringType;

public:
	explicit MinimalStringT(IMinimalAllocator *alloc) :
		m_alloc(alloc), 
		m_str((CharType *)alloc->Allocate(sizeof CharType)), m_msize(1), m_size(0)
	{
		m_str[0] = 0;
	}

	explicit MinimalStringT(ConstStringType str, IMinimalAllocator *alloc) :
		m_alloc(alloc)
	{
		m_size = MinimalString::MinimalStringLength(str);
		m_msize = m_size + 1;
		m_str = (CharType *)alloc->Allocate(m_msize * sizeof CharType);
		MinimalString::MinimalStringCopy(m_str, str);
	}

	~MinimalStringT() {
		m_alloc->Deallocate(m_str);
	}

public:
	void Grow(size_t N)
	{
		if(N <= m_size) return;
		void *p = m_alloc->Reallocate(m_str, N * sizeof CharType);
		m_str = (CharType *)p;
		m_msize = N;
	}

	void Repair()
	{
		m_size = MinimalString::MinimalStringLength(m_str);
	}

	StringType GetRaw()
	{
		return m_str;
	}

	ConstStringType GetRaw() const
	{
		return m_str;
	}

	size_t GetSize()
	{
		return m_size;
	}

	operator StringType()
	{
		return m_str;
	}

	IMinimalAllocator * GetAllocator()
	{
		return m_alloc;
	}

protected:
	IMinimalAllocator *m_alloc;
	StringType m_str;
	size_t m_msize;
	size_t m_size;
};

#ifdef MINIMAL_USE_PROCESSHEAPSTRING
#ifndef MINIMAL_GALLOCATOR
#define MINIMAL_GALLOCATOR
__declspec(selectany)
ProcessHeapAllocator g_allocator;
#endif

template <typename CharType>
class ProcessHeapStringT : public MinimalStringT<CharType>
{
private:
	ProcessHeapStringT<CharType> & operator =(const ProcessHeapStringT<CharType> &str);

public:
	explicit ProcessHeapStringT()
		: MinimalStringT<CharType>(&g_allocator) {}

	explicit ProcessHeapStringT(const CharType *str)
		: MinimalStringT<CharType>(str, &g_allocator) {}

public:
	ProcessHeapStringT<CharType> & operator =(const CharType *str)
	{
		m_size = MinimalString::MinimalStringLength(str);
		Grow(m_size + 1);
		MinimalString::MinimalStringCopy(m_str, str);
		return *this;
	}

	ProcessHeapStringT<CharType> & operator +=(const CharType *str)
	{
		size_t strLen = MinimalString::MinimalStringLength(str);
		Grow(m_size + strLen + 1);
		MinimalString::MinimalStringCopy(m_str + m_size, str);
		m_size += strLen;
		return *this;
	}
};

typedef ProcessHeapStringT<char>    ProcessHeapStringA;
typedef ProcessHeapStringT<wchar_t> ProcessHeapStringW;

#endif

__inline
bool ToUCS2(MinimalStringT<wchar_t> &dest, LPCSTR source)
{
	int inLen = ::lstrlen(source);
	int outLen = ::MultiByteToWideChar(
		CP_ACP, 0, source, inLen, NULL, 0); 
	if(outLen == 0) return false;
	dest.Grow(outLen + 1);
	int convLen = ::MultiByteToWideChar(
		CP_ACP, 0, source, inLen, dest.GetRaw(), outLen);
	if(outLen != convLen) return false;
	dest.GetRaw()[outLen] = 0;
	dest.Repair();
	return true;
}

__inline
bool ToUTF8(MinimalStringT<char> &dest, LPCSTR source)
{
	MinimalStringT<wchar_t> unistr(dest.GetAllocator());
	if(!ToUCS2(unistr, source)) return false;

	int outLen = ::WideCharToMultiByte(
		CP_UTF8, 0,
		unistr, unistr.GetSize(),
		NULL, 0, NULL, NULL); 
	if(outLen == 0) return false;
	dest.Grow(outLen + 1);
	int convLen = ::WideCharToMultiByte(
		CP_UTF8, 0,
		unistr, unistr.GetSize(),
		dest, outLen, NULL, NULL); 
	if(outLen != convLen) return false;
	dest.GetRaw()[outLen] = 0;
	dest.Repair();
	return true;
}

}
